' RoTex(texture:TTexture, angle:Float, Scale:Float = 1)
' CreateGrid:TPixmap(size:Int=512,R:Int = 255, G:Int = 255, B:Int = 255, Alpha:Int = 255)
' CreateFakeShaderTexture:TPixmap()
' CreateSunTexture:TPixmap()
' CreateBrightmap:TTexture(r:Int,g:Int,b:Int)
' CreateBaseTexture:TPixmap(size:Int=32,r:Int=255,g:Int=255,b:Int=255,a:Float=1.0)
' CreateRingTexture:TPixmap(p:TPlanet,size:Int=512,factor:Float=1.0,fader:Float=0.25)



' ------------------------------------------------------------------------------------------------
' Rotate and scale a texture using separate calculations
' ------------------------------------------------------------------------------------------------
Function RoTex(texture:TTexture, angle:Float, Scale:Float = 1)

	Local x:Float
	Local y:Float

	ScaleTexture texture, Scale, Scale
	
	RotateTexture texture, angle
	
	x = Cos(angle) / Scale / 2.0
	y = Sin(angle) / Scale / 2.0
	
	PositionTexture texture, (x - 0.5) - y, (y - 0.5) + x
	
End Function



' ------------------------------------------------------------------------------------------------
' Create a Grid Texture
' ------------------------------------------------------------------------------------------------
Function CreateGrid:TPixmap(Size:Int = 512, R:Int = 255, G:Int = 255, B:Int = 255, Alpha:Int = 255)

	Local pixmap:TPixmap = CreatePixmap(Size, Size, PF_RGBA8888)

	For Local x:Int = 0 To Size - 1
		
		For Local y:Int = 0 To Size - 1
		
			WritePixel(pixmap, x, y, CombineRGBA(R, G, B, Alpha / 50))
		
			If x = 0 Or x = Size - 1 Then WritePixel(pixmap, x, y, CombineRGBA(R, G, B, Alpha))
			If y = 0 Or y = Size - 1 Then WritePixel(pixmap, x, y, CombineRGBA(R, G, B, Alpha))
		
		Next
	
	Next
	
	Return pixmap

End Function



' ------------------------------------------------------------------------------------------------
' Create a Fake Planet Shader Pixmap (pixmap must be loaded with flag 64!)
' ------------------------------------------------------------------------------------------------
Function CreateFakeShaderTexture:TPixmap()
	
	Local pixmap:TPixmap = CreatePixmap(512, 512, PF_RGBA8888)
	
	Local x:Int, y:Int, i:Float, j:Int, col:Int, rgb:Int
		
	For x = 0 To 511
		
		For y = 0 To 511
			
			rgb = 255 * $1000000 + 255 * $10000 + 255 * $100 + 255
			WritePixel(pixmap, x, y, rgb)
			
		Next
		
	Next
	
	For j = 0 To 255
		
		col = j * 1.0 / Exp((255 - j) * 0.001)
		
		If col > 255 Then col = 255
		If col < 0 Then col = 0
		
		rgb = col * $1000000 + col * $10000 + col * $100 + col
		
		For i = 0 To 360 Step 0.1
			
			WritePixel(pixmap, 255 + (Sin(i) * j), 255 + (Cos(i) * j), rgb)
			
		Next
		
	Next

	Return pixmap
	
End Function



' ------------------------------------------------------------------------------------------------
' Creates a Sun Texture
' ------------------------------------------------------------------------------------------------
Function CreateSunTexture:TPixmap()
	
	Local pixmap:TPixmap = CreatePixmap(512, 512, PF_RGBA8888)
	
	Local i:Float, j:Int, col:Int, rgb:Int
	
	For j = 0 To 255
		
		col = 255 - j
		If col > 255 Then col = 255
		rgb = col * $1000000 + col * $10000 + col * $100 + col
		
		For i = 0 To 360 Step 0.1
			
			WritePixel(pixmap, 256 + (Sin(i) * j), 256 + (Cos(i) * j), rgb)
			
		Next
		
	Next
	
	Return pixmap
	
End Function



' ----------------------------------------------------------------------------
' create a micro texture for brightness change
' ----------------------------------------------------------------------------
Function CreateBrightmap:TPixmap(r:Int, g:Int, b:Int)

	Local pixmap:TPixmap = CreatePixmap(2, 2, PF_RGB888)
	Local RGB:Int = CombineRGB(r, g, b)

	WritePixel(pixmap, 0, 0, RGB)
	WritePixel(pixmap, 0, 1, RGB)
	WritePixel(pixmap, 1, 0, RGB)
	WritePixel(pixmap, 1, 1, RGB)

	Return pixmap
	
End Function



' ----------------------------------------------------------------------------
' create a unicolor base texture for coloring a mesh
' ----------------------------------------------------------------------------
Function CreateBaseTexture:TPixmap(Size:Int = 32, r:Int = 255, g:Int = 255, b:Int = 255, a:Float = 1.0)
	
	Local x:Int, y:Int
	Local pixmap:TPixmap = CreatePixmap(Size, Size, PF_RGBA8888)
	
	For x = 0 To Size - 1

		For y = 0 To Size - 1

			WritePixel(pixmap, x, y, CombineRGBA(r, g, b, a * 255))

		Next

	Next
	
	Return pixmap
		
End Function



' ----------------------------------------------------------------------------
' Creates a noisy Ring texture
' ----------------------------------------------------------------------------
Function CreateRingTexture:TPixmap(p:TPlanet, Size:Int = 512, seed:Int = 1, factor:Float = 1.0, fader:Float = 0.25)
	
	Local x:Int, y:Int, col:Int, a:Int
	Local r1:Int, r2:Int, r3:Int, r4:Int, r5:Int
	
	Local pixmap:TPixmap = CreatePixmap(Size, 32, PF_RGBA8888)
	
	For x = 0 To Size - 1
		
		r1 = Normalize(Perlin3D(x * factor, 0, 0, 128, seed, 0, 3), -0.7, 0.7, 64, 192)
		r2 = Normalize(Perlin3D(x * factor, 0, 0, 32, seed, 0, 5), -0.7, 0.7, 96, 255)
		r3 = Normalize(Perlin3D(x * factor, 0, 0, 1, seed, 0, 7), -0.7, 0.7, 0, 64)
		
		r4 = Normalize(x, 0, Size * fader, 0, 255)
		r5 = Normalize(x, Size, Size * fader, 0, 255)
		
		r5 = Filter(r4, r5, "Multiply")
		
		r1 = LimitInt(r1, 0, 255)
		r2 = LimitInt(r2, 0, 255)
		r3 = LimitInt(r3, 0, 255)
		r4 = LimitInt(r4, 0, 255)
		r5 = LimitInt(r5, 0, 255)
		
		col = Filter(r5, r1, "Multiply")
		col = Filter(col, r2, "Softlight")
		col = Filter(col, r3, "Lighten")
		
		a = Filter(r5, r1, "Multiply")
		
		For y = 0 To 31
			
			WritePixel(pixmap, x, y, CombineRGBA(col, col, col, a))
			
		Next
		
	Next
	
	Return pixmap
	
End Function