' Curve:Float(val:Float, mn:Float, mx:Float, typ:Int = 3, amp:Float = 1)
' LinearInterpolate:Float(x1:Float,x2:Float,mu:Float=0.5)
' CosineInterpolate:Float(x1:Float,x2:Float,mu:Float=0.5)
' CubicInterpolate:Float(v0:Float,v1:Float,v2:Float,v3:Float,mu:Float=0.5)
' HermiteInterpolate:Float(y0:Float,y1:Float,y2:Float,y3:Float,t:Float=0.5,tension:Float=0.0,bias:Float=0.0)
' CurveValue:Float(newvalue:Float, oldvalue:Float, increments:Int)



' ----------------------------------------------------------------------------
' Smooth curve Interpolation
' ----------------------------------------------------------------------------
Function Curve:Float(val:Float, mn:Float, mx:Float, typ:Int = 3, amp:Float = 1)

	Local olval:Float
	Local tween:Float
	Local cos1:Float
	Local dif:Float
	
	Val = Val - mn
	mx = mx - mn
	If amp <> 1 Then olval = Val
	tween = ((Val / mx) * 90)
	
	' smooth out
	If typ <= 1 Then
		
		cos1 = Cos(tween - 90)
		Val = cos1 * mx
		
	EndIf
	
	' smooth in
	If typ = 2 Then
		
		cos1 = 1 - Cos((tween))
		Val = cos1 * mx
		
	EndIf
	
	' smooth in and out
	If typ = 3 Then
	
		cos1 = Cos(tween - 90) * Sin(tween)
		Val = cos1 * mx
		
	EndIf
	
	' amplify
	If amp <> 1 Then
	
		dif = olval - Val
		Val = olval - (dif * amp)
		
	EndIf
	
	Return Val + mn

End Function



' ----------------------------------------------------------------------------
' Linear Interpolation
' ----------------------------------------------------------------------------
Function LinearInterpolate:Float(x1:Float, x2:Float, mu:Float = 0.5)
	
	Return (x1 * (1.0 - mu) + x2 * mu)
	
End Function



' ----------------------------------------------------------------------------
' Cosine Interpolation
' ----------------------------------------------------------------------------
Function CosineInterpolate:Float(x1:Float, x2:Float, mu:Float = 0.5)

	Local mu2:Float = 1.0 - ((Cos(mu * 180.0) + 1.0) / 2.0)
	Return (x1 * (1.0 - mu2) + x2 * mu2)
	
End Function



' ----------------------------------------------------------------------------
' Cubic Interpolation
' ----------------------------------------------------------------------------
Function CubicInterpolate:Float(v0:Float, v1:Float, v2:Float, v3:Float, mu:Float = 0.5)
	
	Local P:Float = (v3 - v2) - (v0 - v1)
	Local Q:Float = (v0 - v1) - P
	Local R:Float = v2 - v0
	Local S:Float = v1
	
	Return P * mu ^ 3 + Q * mu ^ 2 + R * mu + S
	
End Function



' ----------------------------------------------------------------------------
' Hermite Interpolation
' ----------------------------------------------------------------------------
Function HermiteInterpolate:Float(y0:Float, y1:Float, y2:Float, y3:Float, t:Float = 0.5, tension:Float = 0.0, bias:Float = 0.0)

	Local t2:Float = t * t
	Local t3:Float = t2 * t
	Local tb:Float = (1 + bias) * (1 - tension) / 2

	Return (2 * t3 - 3 * t2 + 1) * y1 + (((t3 - 2 * t2 + t) * (y1 - y0 + y2 - y1)) + (t3 - t2) * (y2 - y1 + y3 - y2)) * tb + (3 * t2 - 2 * t3) * y2
	
End Function



' ------------------------------------------------------------------------------------------------
' Curve Interpolation
' ------------------------------------------------------------------------------------------------
Function CurveValue:Float(newvalue:Float, oldvalue:Float, increments:Int)

	If increments > 1 Then oldvalue = oldvalue - (oldvalue - newvalue) / increments
	
	If increments <= 1 Then oldvalue = newvalue
	
	Return oldvalue
			
End Function