' Turn(Ent:TEntity, X:Float, Y:Float, Z:Float, Glob:Int = False)
' EulerToQuat:TQuaternion(pitch:Float,yaw:Float,roll:Float)
' QuatToEuler2(x:Float,y:Float,z:Float,w:Float,pitch:Float Var,yaw:Float Var,roll:Float Var)
' MultiplyQuats:TQuaternion(q1:TQuaternion,q2:TQuaternion)
' NormalizeQuat:TQuaternion(q:TQuaternion)



' ------------------------------------------------------------------------------------------------
' Turn Entity using Quaternions
' ------------------------------------------------------------------------------------------------
Function Turn:Float[] (Ent:TEntity, X:Float, Y:Float, Z:Float, Glob:Int = False)

	Local Pitch:Float = 0.0
	Local Yaw:Float = 0.0
	Local Roll:Float = 0.0
	
	Local Quat:TQuaternion = EulerToQuat(0.0, 0.0, 0.0)
	Local Turn_Quat:TQuaternion = EulerToQuat(0.0, 0.0, 0.0)
	
	Local w:Float[] = [0.0, 0.0, 0.0]
	
	If Glob = False
	
		Quat = EulerToQuat(EntityPitch(Ent, True), EntityYaw(Ent, True), EntityRoll(Ent, True))
		Turn_Quat = EulerToQuat(X, Y, Z)
		Quat = MultiplyQuats(Quat, Turn_Quat)
		Quat = NormalizeQuat(Quat)
		
		w = [Quat.X, Quat.Y, Quat.Z]
		
		QuatToEuler2(Quat.x, Quat.y, Quat.z, Quat.w, Pitch, Yaw, Roll)
		RotateEntity Ent, Pitch, Yaw, Roll
				
	Else
	
		RotateEntity Ent, EntityPitch(Ent) + X, EntityYaw(Ent) + Y, EntityRoll(Ent) + Z
		w = [0.0, 0.0, 0.0]
		
	EndIf
	
	Return w
	
End Function



' ------------------------------------------------------------------------------------------------
' Euler to Quaternion
' ------------------------------------------------------------------------------------------------
Function EulerToQuat:TQuaternion(pitch:Float, yaw:Float, roll:Float)

	Local cr:Float = Cos(-roll / 2.0)
	Local cp:Float = Cos(pitch / 2.0)
	Local cy:Float = Cos(yaw / 2.0)
	Local sr:Float = Sin(-roll / 2.0)
	Local sp:Float = Sin(pitch / 2.0)
	Local sy:Float = Sin(yaw / 2.0)
	Local cpcy:Float = cp * cy
	Local spsy:Float = sp * sy
	Local spcy:Float = sp * cy
	Local cpsy:Float = cp * sy
	
	Local q:TQuaternion = New TQuaternion
	
	q.w:Float = cr * cpcy + sr * spsy
	q.x:Float = sr * cpcy - cr * spsy
	q.y:Float = cr * spcy + sr * cpsy
	q.z:Float = cr * cpsy - sr * spcy
	
	Return q
	
End Function



' ------------------------------------------------------------------------------------------------
' Quaternion to Euler
' ------------------------------------------------------------------------------------------------
Function QuatToEuler2(x:Float, y:Float, z:Float, w:Float, pitch:Float Var, yaw:Float Var, roll:Float Var)

	Local QuatToEulerAccuracy:Double = 0.1'1.0 / 2 ^ 31

	Local sint:Float = (2.0 * w * y) - (2.0 * x * z)
	Local cost_temp:Float = 1.0 - (sint * sint)
	Local cost:Float

	If Abs(cost_temp) > QuatToEulerAccuracy

		cost = Sqr(cost_temp)

	Else

		cost = 0.0

	EndIf

	Local sinv:Float, cosv:Float, sinf:Float, cosf:Float
	
	If Abs(cost) > QuatToEulerAccuracy
	
		sinv = ((2.0 * y * z) + (2.0 * w * x)) / cost
		cosv = (1.0 - (2.0 * x * x) - (2.0 * y * y)) / cost
		sinf = ((2.0 * x * y) + (2.0 * w * z)) / cost
		cosf = (1.0 - (2.0 * y * y) - (2.0 * z * z)) / cost
		
	Else
		
		sinv = (2.0 * w * x) - (2.0 * y * z)
		cosv = 1.0 - (2.0 * x * x) - (2.0 * z * z)
		sinf = 0.0
		cosf = 1.0
		
	EndIf
	
	pitch = ATan2(sint, cost)
	yaw = ATan2(sinf, cosf)
	roll = -ATan2(sinv, cosv)
	
End Function



' ------------------------------------------------------------------------------------------------
' Multiply Quaternion
' ------------------------------------------------------------------------------------------------
Function MultiplyQuats:TQuaternion(q1:TQuaternion, q2:TQuaternion)

	Local q:TQuaternion = New TQuaternion
	
	q.w = q1.w * q2.w - q1.x * q2.x - q1.y * q2.y - q1.z * q2.z
	q.x = q1.w * q2.x + q1.x * q2.w + q1.y * q2.z - q1.z * q2.y
	q.y = q1.w * q2.y + q1.y * q2.w + q1.z * q2.x - q1.x * q2.z
	q.z = q1.w * q2.z + q1.z * q2.w + q1.x * q2.y - q1.y * q2.x

	Return q

End Function



' ------------------------------------------------------------------------------------------------
' Normalize Quaternion
' ------------------------------------------------------------------------------------------------
Function NormalizeQuat:TQuaternion(q:TQuaternion)

	Local uv:Float = Sqr(q.w * q.w + q.x * q.x + q.y * q.y + q.z * q.z)

	q.w = q.w / uv
	q.x = q.x / uv
	q.y = q.y / uv
	q.z = q.z / uv

	Return q

End Function