SuperStrict

Import sidesign.minib3d

Graphics3D 1920, 1080, 32, 2

Const STARS:Int = 25000        ' number of stars per cluster part
Const DIMENSION:Int = 2000     ' cluster dimension
Const SPEED:Float = 5.0        ' overall move speed multiplicator
Const BRAKE:Float = 0.95       ' brake multiplicator
Const VSYNC:Int = True         ' Vertical Sync

' debug
Const HOMING:Int = True        ' turn homing on/off       Debug: FALSE
Const COLORED:Int = False      ' colorize clusters        Debug: TRUE
Const Scale:Float = 0.75        ' star scale factor        Debug: 1.0
Const FADE:Int = True          ' fade distant clusters    Debug: False

' ----------------------------------------------------------------------------
' Global variables
' ----------------------------------------------------------------------------
Global localx:Float, localy:Float, localz:Float
Global globalx:Float, globaly:Float, globalz:Float
Global simx:Float, simy:Float, simz:Float
Global xspeed:Float, yspeed:Float, zspeed:Float, rspeed:Float
Global xadd:Float, yadd:Float, zadd:Float, radd:Float, tadd:Float

Global FPS:TFPS = New TFPS

' ----------------------------------------------------------------------------
' Init scene
' ----------------------------------------------------------------------------
Global masterpivot:TPivot = CreatePivot()

Global cam:TCamera = CreateCamera()
CameraRange cam, 1, DIMENSION * 8
PositionEntity cam, 0, 0, 0.01

Local mesh:TMesh = CreateMesh(masterpivot)
Local surf:TSurface = CreateSurface(mesh)
EntityFX mesh, 1 + 2
EntityBlend mesh, 3

Local tex:TTexture = LoadTexture("star.tga", 2)
EntityTexture mesh, tex
TextureBlend tex, 5

' ----------------------------------------------------------------------------
' Add Quads
' ----------------------------------------------------------------------------
Local q:TQuad
Local quadlist:TList = CreateList()

Local lx:Float, ly:Float, lz:Float

For Local i:Int = 1 To STARS

	q = New TQuad
	q.mesh = mesh
	q.surf = surf

	q.x = Rnd(-DIMENSION / 2, DIMENSION / 2)
	q.y = Rnd(-DIMENSION / 2, DIMENSION / 2)
	q.z = Rnd(-DIMENSION / 2, DIMENSION / 2)

	' randomized clustering
	If Rnd(1) > 0.001 Then

		Local d:Float = Rnd(0.001, 0.075)

		q.x = lx + Rnd(-DIMENSION * d, DIMENSION * d)
		q.y = ly + Rnd(-DIMENSION * d, DIMENSION * d)
		q.z = lz + Rnd(-DIMENSION * d, DIMENSION * d)

	EndIf

	q.scalex = Rnd(1, Rnd(2, Rnd(4, Rnd(8, Rnd(16, 32))))) * Scale
	q.scaley = q.scalex

	Local c:Float = (q.scalex / 16.0)
	If Rnd(1) > 0.9 Then c:+Rnd(0.25, 0.75)
	If c >= 0.0 And c < 0.3 Then q.RGB = [255, 160, 128]
	If c >= 0.3 And c < 0.5 Then q.RGB = [255, 192, 128]
	If c >= 0.5 And c < 0.6 Then q.RGB = [255, 255, 128]
	If c >= 0.6 And c < 0.7 Then q.RGB = [255, 255, 255]
	If c >= 0.7 And c < 0.8 Then q.RGB = [128, 192, 255]
	If c >= 0.9 And c <= 1.0 Then q.RGB = [128, 128, 255]

	q.Add()
	q.Update(cam)

	ListAddLast(quadlist, q)
	
	lx = q.x
	ly = q.y
	lz = q.z

Next

' ----------------------------------------------------------------------------
' Duplicate Quad mesh in a 9x9x9 matrix
' ----------------------------------------------------------------------------
Local meshes:TEntity[9, 9, 9]
Local meshlist:TList = CreateList()

For Local x:Int = 0 To 4 Step 1

	For Local y:Int = 0 To 4 Step 1

		For Local z:Int = 0 To 4 Step 1

			meshes[x, y, z] = CopyEntity(mesh)

			PositionEntity meshes[x, y, z], (x - 2) * DIMENSION, (y - 2) * DIMENSION, (z - 2) * DIMENSION

			If COLORED Then

				EntityColor meshes[x, y, z], Rand(128, 255), Rand(128, 255), Rand(128, 255)

			Else

				EntityColor meshes[x, y, z], 255, 255, 255

			EndIf

			ListAddLast(meshlist, meshes[x, y, z])

		Next

	Next

Next

HideEntity mesh

MoveMouse(GraphicsWidth() / 2, GraphicsHeight() / 2)

' ----------------------------------------------------------------------------
' Main Loop
' ----------------------------------------------------------------------------
While Not AppTerminate()

	If KeyHit(KEY_ESCAPE) Then End

	' axis rotation
	Local pitch:Float = Normalize(MouseY(), 0, GraphicsHeight() - 1, -2, 2)
	Local yaw:Float = Normalize(MouseX(), 0, GraphicsWidth() - 1, 2, -2)
	Local roll:Float = KeyDown(KEY_Q) - KeyDown(KEY_E)

	' XYZ movement
	Local xmove:Int = KeyDown(KEY_D) - KeyDown(KEY_A)
	Local ymove:Int = KeyDown(KEY_R) - KeyDown(KEY_F)
	Local zmove:Int = KeyDown(KEY_W) - KeyDown(KEY_S)
	Local turbo:Int = KeyDown(KEY_TAB)

	If xmove = 0 And xadd <> 0 Then xadd = 0 Else If xmove And xadd = 0 Then xadd = 0.001
	If ymove = 0 And yadd <> 0 Then yadd = 0 Else If ymove And yadd = 0 Then yadd = 0.001
	If zmove = 0 And zadd <> 0 Then zadd = 0 Else If zmove And zadd = 0 Then zadd = 0.001
	If roll = 0 And radd <> 0 Then radd = 0 Else If roll And radd = 0 Then radd = 0.01
	If turbo = 0 And tadd <> 0 Then tadd:*0.9 Else If turbo And tadd = 0 Then tadd = 0.1

	If tadd < 0.01 Then tadd = 0.0

	If Abs(xmove) Then xadd:+0.01
	If Abs(ymove) Then yadd:+0.01
	If Abs(zmove) Then zadd:+0.01
	If Abs(roll) Then radd:+0.025
	If Abs(turbo) Then tadd:+0.05

	xadd:*1.025
	yadd:*1.025
	zadd:*1.025
	radd:*1.025
	tadd:*1.025

	If xadd > 2.0 Then xadd = 2.0
	If yadd > 2.0 Then yadd = 2.0
	If zadd > 2.0 Then zadd = 2.0
	If radd > 1.5 Then radd = 1.5
	If tadd > 25.0 Then tadd = 25.0

	' calculate movespeed/rollspeed
	If Abs(xmove) Then xspeed = xadd * SPEED * Sgn(xmove) * (1.0 + tadd)
	If Abs(ymove) Then yspeed = yadd * SPEED * Sgn(ymove) * (1.0 + tadd)
	If Abs(zmove) Then zspeed = zadd * SPEED * Sgn(zmove) * (1.0 + tadd)
	If Abs(roll) Then rspeed = radd * Sgn(roll)

	' perform rotation and motion
	Turn(cam, pitch, yaw, rspeed)
	MoveEntity cam, xspeed, yspeed, zspeed

	' keep camera at home
	If HOMING Then Home(cam, masterpivot, DIMENSION)

	' only the master instance quads are facing to the camera
	For Local q:TQuad = EachIn quadlist

		q.Update(cam)

	Next

	' fade mesh according to its distance
	If FADE Then

		For Local e:TEntity = EachIn meshlist
	
			Local d:Float = EntityDistance(cam, e)
			Local a:Float = Normalize(d, DIMENSION * 0.5, DIMENSION * 3, 1, 0)
			EntityAlpha e, a
	
		Next

	EndIf

	RenderWorld
	
	FPS.Update()

	BeginMax2D()

		'DrawText(FPS.FPS, 0, 0)
		DrawText("Camera XYZ: "+entityx(cam,1)+" | "+entityy(cam,1)+" | "+entityz(cam,1),0,0)
		DrawText("Global XYZ: "+globalx+" | "+globaly+" | "+globalz,0,15)
		DrawText("Local  XYZ: "+localx+" | "+localy+" | "+localz,0,30)
		DrawText("Sim    XYZ: "+simx+" | "+simy+" | "+simz,0,45)
		

	EndMax2D()

	Flip VSYNC

	' decrease speed
	xspeed:*BRAKE
	yspeed:*BRAKE
	zspeed:*BRAKE
	rspeed:*BRAKE

Wend

End

' keeps the player in a given distance While the world can Move far away
Function Home(target:TEntity, world:TEntity, homesize:Int = 100)

	' store Local player Position
	Local localx:Float = EntityX(target)
	Local localy:Float = EntityY(target)
	Local localz:Float = EntityZ(target)

	' Check X axis
	While localx > homesize
	
		globalx:+homesize
		localx:-homesize
		PositionEntity target, localx, localy, localz
		MoveEntity world, -homesize, 0, 0
		
	Wend

	While localx < - homesize
	
		globalx:-homesize
		localx:+homesize
		PositionEntity target, localx, localy, localz
		MoveEntity world, homesize, 0, 0

	Wend

	' Check Y axis
	While localy > homesize

		globaly:+homesize
		localy:-homesize
		PositionEntity target, localx, localy, localz
		MoveEntity world, 0, -homesize, 0
		
	Wend

	While localy < - homesize

		globaly:-homesize
		localy:+homesize
		PositionEntity target, localx, localy, localz
		MoveEntity world, 0, homesize, 0
		
	Wend

	' Check Z axis
	While localz > homesize
	
		globalz:+homesize
		localz:-homesize
		PositionEntity target, localx, localy, localz
		MoveEntity world, 0, 0, -homesize
		
	Wend

	While localz < - homesize
	
		globalz:-homesize
		localz:+homesize
		PositionEntity target, localx, localy, localz
		MoveEntity world, 0, 0, homesize
		
	Wend

	' store simulated player Position
	simx = localx + globalx
	simy = localy + globaly
	simz = localz + globalz

End Function

' ----------------------------------------------------------------------------
' Normalizes a value to given range
' ----------------------------------------------------------------------------
Function Normalize:Float(value:Float, vmin:Float, vmax:Float, nmin:Float, nmax:Float)

	Return((value - vmin) / (vmax - vmin)) * (nmax - nmin) + nmin

End Function

' ----------------------------------------------------------------------------
' Turn Entity using Quaternions
' ----------------------------------------------------------------------------
Function Turn(Ent:TEntity, X:Float = 0.0, Y:Float = 0.0, Z:Float = 0.0, Glob:Int = False)

	Local Pitch:Float = 0.0
	Local Yaw:Float = 0.0
	Local Roll:Float = 0.0

	Local Quat:TQuaternion = EulerToQuat(0.0, 0.0, 0.0)
	Local Turn_Quat:TQuaternion = EulerToQuat(0.0, 0.0, 0.0)

	If Glob = False

		Quat = EulerToQuat(EntityPitch(ent, True), EntityYaw(ent, True), EntityRoll(ent, True))
		Turn_Quat = EulerToQuat(X, Y, Z)
		Quat = MultiplyQuats(Quat, Turn_Quat)
		Quat = NormalizeQuat(Quat)
		QuatToEuler2(Quat.X, Quat.Y, Quat.Z, Quat.w, Pitch, Yaw, Roll)
		RotateEntity Ent, Pitch, Yaw, Roll

	Else

		RotateEntity Ent, EntityPitch(Ent) + X, EntityYaw(Ent) + Y, EntityRoll(Ent) + Z

	EndIf

End Function

' ----------------------------------------------------------------------------
' Euler to Quaternion
' ----------------------------------------------------------------------------
Function EulerToQuat:TQuaternion(pitch:Float, yaw:Float, roll:Float)

	Local cr:Float = Cos(-roll / 2.0)
	Local cp:Float = Cos(pitch / 2.0)
	Local cy:Float = Cos(yaw / 2.0)
	Local sr:Float = Sin(-roll / 2.0)
	Local sp:Float = Sin(pitch / 2.0)
	Local sy:Float = Sin(yaw / 2.0)
	Local cpcy:Float = cp * cy
	Local spsy:Float = sp * sy
	Local spcy:Float = sp * cy
	Local cpsy:Float = cp * sy

	Local q:TQuaternion = New TQuaternion

	q.w = cr * cpcy + sr * spsy
	q.x = sr * cpcy - cr * spsy
	q.y = cr * spcy + sr * cpsy
	q.z = cr * cpsy - sr * spcy

	Return q

End Function

' ----------------------------------------------------------------------------
' Quaternion to Euler
' ----------------------------------------------------------------------------
Function QuatToEuler2(x:Float, y:Float, z:Float, w:Float, pitch:Float Var, yaw:Float Var, roll:Float Var)

	Local QuatToEulerAccuracy:Double = 1.0 / 2 ^ 31

	Local sint:Float = (2.0 * w * y) - (2.0 * x * z)
	Local cost_temp:Float = 1.0 - (sint * sint)
	Local cost:Float

	If Abs(cost_temp) > QuatToEulerAccuracy

		cost = Sqr(cost_temp)

	Else

		cost = 0.0

	EndIf

	Local sinv:Float, cosv:Float, SinF:Float, CosF:Float

	If Abs(cost) > QuatToEulerAccuracy

		sinv = ((2.0 * y * z) + (2.0 * w * x)) / cost
		cosv = (1.0 - (2.0 * x * x) - (2.0 * y * y)) / cost
		SinF = ((2.0 * x * y) + (2.0 * w * z)) / cost
		CosF = (1.0 - (2.0 * y * y) - (2.0 * z * z)) / cost

	Else

		sinv = (2.0 * w * x) - (2.0 * y * z)
		cosv = 1.0 - (2.0 * x * x) - (2.0 * z * z)
		SinF = 0.0
		CosF = 1.0

	EndIf

	pitch = ATan2(sint, cost)
	yaw = ATan2(SinF, CosF)
	roll = -ATan2(sinv, cosv)

End Function

' ----------------------------------------------------------------------------
' Multiply Quaternion
' ----------------------------------------------------------------------------
Function MultiplyQuats:TQuaternion(q1:TQuaternion, q2:TQuaternion)

	Local q:TQuaternion = New TQuaternion
	
	q.w = q1.w * q2.w - q1.x * q2.x - q1.y * q2.y - q1.z * q2.z
	q.x = q1.w * q2.x + q1.x * q2.w + q1.y * q2.z - q1.z * q2.y
	q.y = q1.w * q2.y + q1.y * q2.w + q1.z * q2.x - q1.x * q2.z
	q.z = q1.w * q2.z + q1.z * q2.w + q1.x * q2.y - q1.y * q2.x

	Return q

End Function

' ----------------------------------------------------------------------------
' Normalize Quaternion
' ----------------------------------------------------------------------------
Function NormalizeQuat:TQuaternion(q:TQuaternion)

	Local uv:Float = Sqr(q.w * q.w + q.x * q.x + q.y * q.y + q.z * q.z)

	q.w = q.w / uv
	q.x = q.x / uv
	q.y = q.y / uv
	q.z = q.z / uv

	Return q

End Function

' ------------------------------------------------------------------------------------------------
' Quad Type
' ------------------------------------------------------------------------------------------------
Type TQuad

	Field x:Float = 0.0										' position x
	Field y:Float = 0.0										' position y
	Field z:Float = 0.0										' position z

	Field row:Int = 0										' texture row
	Field col:Int = 0										' texture column

	Field scalex:Float = 1.0								' current size X
	Field scaley:Float = 1.0								' current size Y

	Field RGB:Int[] = [255, 255, 255]						' Vertex Color
	Field Alpha:Float = 1.0									' Vertex Alpha

	Field v:Int = 0											' vertex counter
	Field mesh:TMesh = Null									' mesh pointer
	Field surf:TSurface = Null								' surface pointer

	' --------------------------------------------------------------------------------------------
	' METHOD: Add new Quad
	' --------------------------------------------------------------------------------------------
	Method Add(col:Int = 0, row:Int = 0)

		' surface setup
		If surf = Null Then surf = surf
		If v + 4 > 32768 Then
			v = 0
			surf = CreateSurface(mesh)
			Print "New Surface"
		EndIf

		' add Vertices
		Local V0:Int = AddVertex(surf, 0, 0, 0, 1, 0)
		Local V1:Int = AddVertex(surf, 0, 0, 0, 1, 1)
		Local V2:Int = AddVertex(surf, 0, 0, 0, 0, 1)
		Local V3:Int = AddVertex(surf, 0, 0, 0, 0, 0)

		' color vertices
		VertexColor surf, V0, RGB[0], RGB[1], RGB[2], Alpha
		VertexColor surf, V1, RGB[0], RGB[1], RGB[2], Alpha
		VertexColor surf, V2, RGB[0], RGB[1], RGB[2], Alpha
		VertexColor surf, V3, RGB[0], RGB[1], RGB[2], Alpha

		' connect triangles
		AddTriangle surf, V0, V1, V2
		AddTriangle surf, V0, V2, V3

		VertexTexCoords surf, V0, col + 1, row
		VertexTexCoords surf, V1, col + 1, row + 1
		VertexTexCoords surf, V2, col, row + 1
		VertexTexCoords surf, V3, col, row

		' increase vertex counter
		If v >= 4 Then v = V0 + 4 Else v = V0

		If v Mod 100000 = 0 Then Print v

	End Method

	' --------------------------------------------------------------------------------------------
	' METHOD: Update a Quad
	' --------------------------------------------------------------------------------------------
	Method Update(target:TEntity)

		TFormVector scalex, 0, 0, target, Null
		Local X1:Float = TFormedX()
		Local Y1:Float = TFormedY()
		Local Z1:Float = TFormedZ()

		TFormVector 0, scaley, 0, target, Null
		Local X2:Float = TFormedX()
		Local Y2:Float = TFormedY()
		Local Z2:Float = TFormedZ()

		' set vertices
		VertexCoords surf, v + 0, x - x1 - x2, y - y1 - y2, z - z1 - z2
		VertexCoords surf, v + 1, x - x1 + x2, y - y1 + y2, z - z1 + z2
		VertexCoords surf, v + 2, x + x1 + x2, y + y1 + y2, z + z1 + z2
		VertexCoords surf, v + 3, x + x1 - x2, y + y1 - y2, z + z1 - z2

	End Method

End Type

' ------------------------------------------------------------------------------------------------
' Exact FPS counter
' ------------------------------------------------------------------------------------------------

Type TFPS

	Global renders:Int = 0

	Field FPS:Int = 60										' current FPS value
	Field interval:Int = 999
	Field averagefps:Int = 60								' average FPS value
	Field framedrops:Int = 0								' dropped frames
	Field memory:Int = 0									' used memory
	Field oldlooptime:Int = 0
	Field looptime:Int = 0
	Field multi:Float										' main loop multiplicator

	Field old:Int = MilliSecs()
	Field totalfps:Int = 30
	Field Ticks:Int

	' --------------------------------------------------------------------------------------------
	' METHOD: Update FPS counter
	' --------------------------------------------------------------------------------------------
	Method Update()

		renders:+1

		looptime = MilliSecs() - oldlooptime
		oldlooptime = MilliSecs()

		If MilliSecs() - old >= interval Then

			old = MilliSecs()

			FPS = renders * (1000 / interval)
			renders = 0
			totalfps:+FPS
			Ticks:+1
			
			averagefps = totalfps / Ticks

			memory = GCMemAlloced()

		EndIf

	End Method

End Type